(function ($, document) {

	var websiteId = null;
	var establishmentsByCityCode = {};
	var validAirports = {};

	var translations = {};

	// Domain currently ignored
	function translate(key, domain) {

		if (translations.hasOwnProperty(key)) {
			return translations[key];
		}

		return key;
	}

	function loadTranslations() {
		translations = JSON.parse($('#br_searchbox_translations').val());
	}

	$(document).ready(function () {

		websiteId = $("#br_searchbox_website_id").val();

		loadTranslations();

		var today = new Date();
		var tomorrow = new Date(today);
		tomorrow.setDate(tomorrow.getDate() + 1);

		var checkinElement = document.getElementById("br_searchbox_checkin");
		var checkoutElement = document.getElementById("br_searchbox_checkout");
		var todayString = today.toISOString().split("T")[0];
		var tomorrowString = tomorrow.toISOString().split("T")[0];

		checkinElement.value = todayString;
		checkoutElement.value = tomorrowString;
		checkinElement.min = todayString;
		checkoutElement.min = tomorrowString;

		fetchExternalData();
		$('#br_searchbox_submit').click(buttonSubmit);
		$('#br_searchbox_destination').change(onChangeDestination);
		$('#br_searchbox_count_rooms').change(onChangeCountRooms);

		// When checkin or checkout are changed, make sure checkin < checkout
		$('#br_searchbox_checkin').change(onChangeCheckinCheckout);
		$('#br_searchbox_checkout').change(onChangeCheckinCheckout);

		if ($("#br_searchbox_enable_flights").val() === "true") {
			onChangeWithFlight();
			$('#br_searchbox_with_flight').change(onChangeWithFlight);
		}

	});

	function onChangeCheckinCheckout(evt) {
		var checkinElement = document.getElementById("br_searchbox_checkin");
		var checkoutElement = document.getElementById("br_searchbox_checkout");
		var checkinVal = $(checkinElement).val();
		var checkoutVal = $(checkoutElement).val();

		// Checkin changed and checkout is before checkin
		if (evt.target == checkinElement && checkoutVal <= checkinVal) {
			var dayAfter = new Date(checkinVal);
			dayAfter.setDate(dayAfter.getDate() + 1);
			checkoutElement.value = dayAfter.toISOString().split("T")[0];
		}

		// Checkout changed and checkin is after checkout
		if (evt.target == checkoutElement && checkoutVal <= checkinVal) {
			var dayBefore = new Date(checkoutVal);
			dayBefore.setDate(dayBefore.getDate() - 1);
			checkinElement.value = dayBefore.toISOString().split("T")[0];
		}
	}

	function getPassengerGroups() {
		var roomCount = $("#br_searchbox_count_rooms").val();

		var result = [];
		for (var i = 0; i < parseInt(roomCount); i++) {

			var adultCount = $('#br_searchbox_paxgroup_' + i + '_adults').val();
			var childCount = $('#br_searchbox_paxgroup_' + i + '_children').val();
			var childrenAges = [];

			for (var j = 0; j < parseInt(childCount); j++) {
				childrenAges.push($('#br_searchbox_paxgroup_' + i + '_children_age_' + j).val())
			}

			result.push({ adults: adultCount, childrenAges: childrenAges });
		}

		return result;

	}

	function onChangeWithFlight() {
		var withFlight = $("#br_searchbox_with_flight").is(":checked");
		if (withFlight) {
			$("#br_searchbox_origin_container").show();
		}
		else {
			$("#br_searchbox_origin_container").hide();
		}
	}

	function onChangeCountRooms() {
		var roomCount = $("#br_searchbox_count_rooms").val();
		$("#br_searchbox_rooms_container").empty();

		for (var i = 0; i < parseInt(roomCount); i++) {

			var roomHeader =
				$(document.createElement('h4'))
					.text(translate('Room', 'busyrooms') + ' #' + i);

			var adultWrapper =
				$(document.createElement('div'))
					.addClass('br_searchbox_paxgroup_adults_container');

			var childWrapper =
				$(document.createElement('div'))
					.addClass('br_searchbox_paxgroup_children_container');

			var adultCountElemLabel =
				$(document.createElement('label'))
					.attr('for', 'br_searchbox_paxgroup_' + i + '_adults')
					.text(translate("Number of adults", "busyrooms"));

			var adultCountElem =
				$(document.createElement('input'))
					.attr('id', 'br_searchbox_paxgroup_' + i + '_adults')
					.attr('type', 'number')
					.val(1)
					.attr('min', '1')
					.attr('max', $('#br_searchbox_max_adults_per_room').val());

			var childCountElemLabel =
				$(document.createElement('label'))
					.attr('for', 'br_searchbox_paxgroup_' + i + '_children')
					.text(translate("Number of children", "busyrooms"));

			var childCountElem =
				$(document.createElement('input'))
					.attr('id', 'br_searchbox_paxgroup_' + i + '_children')
					.change({ room_index: i }, onChildCountChange)
					.val(0)
					.attr('type', 'number')
					.attr('min', '0')
					.attr('max', $('#br_searchbox_max_children_per_room').val());

			var childAgeContainer =
				$(document.createElement('div'))
					.addClass('br_searchbox_paxgroup_child_age_container')
					.attr('id', 'br_searchbox_paxgroup_' + i + '_children_age_container');

			var paxGroup =
				$(document.createElement('div'))
					.addClass('br_searchbox_pax_group');

			adultCountElemLabel.appendTo(adultWrapper);
			adultCountElem.appendTo(adultWrapper);
			childCountElemLabel.appendTo(childWrapper);
			childCountElem.appendTo(childWrapper);

			roomHeader.appendTo(paxGroup);
			adultWrapper.appendTo(paxGroup);
			childWrapper.appendTo(paxGroup);
			childAgeContainer.appendTo(paxGroup);

			paxGroup.appendTo($("#br_searchbox_rooms_container"));

			$('#br_searchbox_paxgroup_' + i + '_children').trigger('change');
		}
	}

	function onChildCountChange(evt) {

		var target = $(evt.target);
		var childCount = target.val();
		var roomIndex = evt.data.room_index;
		var container = $('#br_searchbox_paxgroup_' + roomIndex + '_children_age_container');

		container.empty();

		if (childCount > 0) {
			container.show();
		}
		else {
			container.hide();
			return;
		}

		$(document.createElement('h5'))
			.text(translate('Children ages', "busyrooms"))
			.appendTo(container);

		var inputContainer =
			$(document.createElement('div'))
				.addClass('br_searchbox_paxgroup_children_age_container');

		for (var i = 0; i < childCount; i++) {

			$(document.createElement('input'))
				.attr('id', 'br_searchbox_paxgroup_' + roomIndex + '_children_age_' + i)
				.val(0)
				.attr('type', 'number')
				.attr('min', '0')
				.attr('max', $('#br_searchbox_max_child_age').val())
				.appendTo(inputContainer);
		}

		inputContainer.appendTo(container);
	}

	function fetchExternalData() {
		$.ajax({
			url: "http://api.direct-reservation.net/" + websiteId + "/establishmentLocations?rnd=" + Math.random() + "&jsoncallback=?",
			dataType: "jsonp",
			success: function (data) {
				bindEstablishmentLocations(data.establishmentLocations);
			}
		});

		$.ajax({
			url: "http://api.direct-reservation.net/en/airports?" + "rnd=" + Math.random() + "&jsoncallback=?",
			dataType: "jsonp",
			success: function (data) {
				bindAirports(data.airports);
			}
		})
	}

	function onChangeDestination() {

		var cityCode = $("#br_searchbox_destination").val();
		$("#br_searchbox_establishment").empty();

		$(document.createElement('option'))
			.text(translate("All establishments", "busyrooms"))
			.val("all")
			.appendTo($("#br_searchbox_establishment"));

		establishmentsByCityCode[cityCode].forEach(function (establishment) {
			$(document.createElement('option'))
				.text(establishment.Name)
				.val(establishment.ID)
				.appendTo($("#br_searchbox_establishment"));
		});
	}

	function bindAirports(airports) {

		var airportsFormatted = [];
		for (var i = 0; i < airports.length; i++) {
			var airport = airports[i];
			validAirports[airport.Code] = true;
			airportsFormatted.push({
				label: airport.Code + ", " + airport.Name,
				value: airport.Code
			});
		}

		$("#br_searchbox_origin").autocomplete({
			source: airportsFormatted
		});

	}

	function bindEstablishmentLocations(establishmentLocations) {
		establishmentLocations.forEach(function (location) {

			establishmentsByCityCode[location.CityCode] = [];

			location.Estbalishments.forEach(function (establishment) {
				establishmentsByCityCode[location.CityCode].push(establishment);
			});

			$(document.createElement('option'))
				.text(location.City + ", " + location.Country)
				.val(location.CityCode)
				.appendTo($("#br_searchbox_destination"));
		});

		onChangeDestination();
		onChangeCountRooms();
	}

	function ValidateForm() {

		$('.br_searchbox_error').removeClass('br_searchbox_error');

		// Validate airport
		var selectedAirport = $('#br_searchbox_origin').val();
		if (!validAirports.hasOwnProperty(selectedAirport)) {
			$('#br_searchbox_origin').addClass('br_searchbox_error');
			return false;
		}


		return true;

	}

	function buttonSubmit(evt) {

		if (!ValidateForm()) {
			return;
		}

		var establishmentIds;
		var selectedEstablishmentId = $("#br_searchbox_establishment").val();
		var selectedCityCode = $("#br_searchbox_destination").val();

		if (selectedEstablishmentId == "all") {
			establishmentIds = [];
			for (var i = 0; i < establishmentsByCityCode[selectedCityCode].length; i++) {
				establishmentIds.push(establishmentsByCityCode[selectedCityCode][i].ID);
			}
		}
		else {
			establishmentIds = [selectedEstablishmentId];
		}

		var languageCode = $("#br_searchbox_blog_language").val().substring(0, 2);

		var query = {
			DateFrom: $("#br_searchbox_checkin").val(),
			DateTo: $("#br_searchbox_checkout").val(),

			Accommodation: {
				EstablishmentIDs: establishmentIds,
			},

			LanguageCode: languageCode,
			PassengerGroups: getPassengerGroups()

		};

		var modules = ['Accommodation'];

		if ($("#br_searchbox_enable_flights").val() === "true") {
			var withFlight = $("#br_searchbox_with_flight").is(":checked");
			if (withFlight) {
				query.Flights = {
					Origin: $("#br_searchbox_origin").val(),
					Destination: $('#br_searchbox_destination').val(),
					OriginName: $("#br_searchbox_origin option:selected").text()
				};
				modules.push('Flights');
			}
		}

		var uri = 'https://api.direct-reservation.net/';

		uri = uri.concat(
			'?websiteID=', websiteId,
			'&query=', encodeURI(JSON.stringify(query)),
			'&modules=', encodeURI(JSON.stringify(modules))
		);

		window.open(uri, '_blank');
	}

}(jQuery, document));